/**
  ******************************************************************************
  * @file    startup.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    14-July-2016
  * @brief   Entry point, fault handlers and IVT
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  *
  ******************************************************************************  
  */ 

#include "fw.h"

/* ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// The entry point for the application.
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////// */
extern int main(void);

/* ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Reserve space for the system stack. This is the stack that the startup code runs on until the application entry point is
// called. When the boot loader is in standalone mode, the interrupt handler that processes the host communication also uses
// this stack. Make sure that the stack never overflows!
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////// */

/* ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// The following are variables created by the linker, indicating where the
// "data" and "bss" segments reside in memory.  The initializers for the
// "data" segment resides immediately following the "text" segment.
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////// */
extern unsigned long _sidata;
extern unsigned long _sdata;
extern unsigned long _edata;
extern unsigned long _sbss;
extern unsigned long _ebss;
extern unsigned long _ramend;

extern void PendSV_Handler(void);
extern void SVC_Handler(void);
extern void SysTick_Handler(void);
extern void SystemInit(void);
extern void DispatchISRToBL(void);

/** @brief "Reset Interrupt Handler"
 *
 *  Due to the resident bootloader architecture, this looks like the reset
 *  Interrupt vector to the application, but it is called after the bootloader has 
 *  started up and verified the image integrity.
 * 
 *  @return Does not return unless main() returns
 */

static void ResetISR(void)
{
    unsigned    long    *a_Src,*a_Dst;
	  __asm volatile(
					" cpsid i				\n" /* Globally enable interrupts. */);
    // Copy the data segment initializers from flash to SRAM.
    a_Src = &_sidata;
    for(a_Dst = &_sdata; a_Dst < &_edata; )
    {
        *a_Dst++ = *a_Src++;
    }
    // Zero fill the bss segment.
    for(a_Dst = &_sbss; a_Dst < &_ebss; )
    {
        *a_Dst++ = 0;
    }
    // In a real architecture, at this point we will set up everything that the bootloader hasn't. 
//    SystemInit();
    // Call the application's entry point. An application will frequently call the startup entry point of the C runtime
    // initialization, ie __start which will eventually call main(). But we are in the boot loader here, so we can not
    // afford to have additional code that steals code space and does things we do not have control over.
    main();
}

__attribute__ ((section(".isr_vector_fw")))
IVTVector g_pfnVectorsFW[IVTENTRYCT] =
{
    // this layout of the IVT corresponds to the STM32F4xx family of controllers.
    (IVTVector)(unsigned long)&_ramend,                      // initial stack pointer          @0x0000_0000
    ResetISR,                                                // reset handler                  @0x0000_0004
    DispatchISRToBL,                                         // NMI handler                    @0x0000_0008
    DispatchISRToBL,                                         // hard fault handler             @0x0000_000C
    DispatchISRToBL,                                         // MPU fault handler              @0x0000_0010
    DispatchISRToBL,                                         // bus fault handler              @0x0000_0014
    DispatchISRToBL,                                         // usage fault handler            @0x0000_0018
    DispatchISRToBL,                                         // Reserved                       @0x0000_001C
    DispatchISRToBL,                                         // Reserved                       @0x0000_0020
    DispatchISRToBL,                                         // Reserved                       @0x0000_0024
    DispatchISRToBL,                                         // Reserved                       @0x0000_0028
    SVC_Handler,                                             // SVCall handler                 @0x0000_002C
    DispatchISRToBL,                                         // Debug monitor handler          @0x0000_0030
    DispatchISRToBL,                                         // Reserved                       @0x0000_0034
    PendSV_Handler,                                          // PendSV handler                 @0x0000_0038
    SysTick_Handler,                                         // SysTick handler                @0x0000_003C
    /* External Interrupts */
    DispatchISRToBL,                                         // WWDG_IRQHandler                @0x0000_0040                                    
    DispatchISRToBL,                                         // PVD_IRQHandler                 @0x0000_0044            
    DispatchISRToBL,                                         // TAMP_STAMP_IRQHandler          @0x0000_0048            
    DispatchISRToBL,                                         // RTC_WKUP_IRQHandler            @0x0000_004C           
    DispatchISRToBL,                                         // FLASH_IRQHandler               @0x0000_0050                           
    DispatchISRToBL,                                         // RCC_IRQHandler                 @0x0000_0054                             
    DispatchISRToBL,                                         // EXTI0_IRQHandler               @0x0000_0058         
    DispatchISRToBL,                                         //                                @0x0000_005C
    DispatchISRToBL,                                         // EXTI2_IRQHandler               @0x0000_0060           
    DispatchISRToBL,                                         // EXTI3_IRQHandler               @0x0000_0064           
    DispatchISRToBL,                                         // EXTI4_IRQHandler               @0x0000_0068           
    DispatchISRToBL,                                         // DMA1_Stream0_IRQHandler        @0x0000_006C   
    DispatchISRToBL,                                         // DMA1_Stream1_IRQHandler        @0x0000_0070    
    DispatchISRToBL,                                         // DMA1_Stream2_IRQHandler        @0x0000_0074    
    DispatchISRToBL,                                         // DMA1_Stream3_IRQHandler        @0x0000_0078    
    DispatchISRToBL,                                         // DMA1_Stream4_IRQHandler        @0x0000_007C    
    DispatchISRToBL,                                         // DMA1_Stream5_IRQHandler        @0x0000_0080    
    DispatchISRToBL,                                         // DMA1_Stream6_IRQHandler        @0x0000_0084    
    DispatchISRToBL,                                         // ADC_IRQHandler                 @0x0000_0088    
    DispatchISRToBL,                                         // CAN1_TX_IRQHandler             @0x0000_008C          
    DispatchISRToBL,                                         // CAN1_RX0_IRQHandler            @0x0000_0090           
    DispatchISRToBL,                                         // CAN1_RX1_IRQHandler            @0x0000_0094           
    DispatchISRToBL,                                         // CAN1_SCE_IRQHandler            @0x0000_0098           
    DispatchISRToBL,                                         // EXTI9_5_IRQHandler             @0x0000_009C           
    DispatchISRToBL,                                         // TIM1_BRK_TIM9_IRQHandler       @0x0000_00A0 
    DispatchISRToBL,                                         // TIM1_UP_TIM10_IRQHandler       @0x0000_00A4 
    DispatchISRToBL,                                         // TIM1_TRG_COM_TIM11_IRQHandler  @0x0000_00A8 
    DispatchISRToBL,                                         // TIM1_CC_IRQHandler             @0x0000_00AC           
    DispatchISRToBL,                                         // TIM2_IRQHandler                @0x0000_00B0    
    DispatchISRToBL,                                         // TIM3_IRQHandler                @0x0000_00B4    
    DispatchISRToBL,                                         // TIM4_IRQHandler                @0x0000_00B8    
    DispatchISRToBL,                                         // I2C1_EV_IRQHandler             @0x0000_00BC           
    DispatchISRToBL,                                         // I2C1_ER_IRQHandler             @0x0000_00C0           
    DispatchISRToBL,                                         // I2C2_EV_IRQHandler             @0x0000_00C4           
    DispatchISRToBL,                                         // I2C2_ER_IRQHandler             @0x0000_00C8             
    DispatchISRToBL,                                         // SPI1_IRQHandler                @0x0000_00CC    
    DispatchISRToBL,                                         // SPI2_IRQHandler                @0x0000_00D0    
    DispatchISRToBL,                                         // USART1_IRQHandler              @0x0000_00D4    
    DispatchISRToBL,                                         // USART2_IRQHandler              @0x0000_00D8    
    DispatchISRToBL,                                         // USART3_IRQHandler              @0x0000_00DC    
    DispatchISRToBL,                                         // EXTI15_10_IRQHandler           @0x0000_00E0           
    DispatchISRToBL,                                         // RTC_Alarm_IRQHandler           @0x0000_00E4           
    DispatchISRToBL,                                         // OTG_FS_WKUP_IRQHandler         @0x0000_00E8               
    DispatchISRToBL,                                         // TIM8_BRK_TIM12_IRQHandler      @0x0000_00EC 
    DispatchISRToBL,                                         // TIM8_UP_TIM13_IRQHandler       @0x0000_00F0 
    DispatchISRToBL,                                         // TIM8_TRG_COM_TIM14_IRQHandler  @0x0000_00F4 
    DispatchISRToBL,                                         // TIM8_CC_IRQHandler             @0x0000_00F8           
    DispatchISRToBL,                                         // DMA1_Stream7_IRQHandler        @0x0000_00FC           
    DispatchISRToBL,                                         // FSMC_IRQHandler                @0x0000_0100    
    DispatchISRToBL,                                         // SDIO_IRQHandler                @0x0000_0104    
    DispatchISRToBL,                                         // TIM5_IRQHandler                @0x0000_0108    
    DispatchISRToBL,                                         // SPI3_IRQHandler                @0x0000_010C    
    DispatchISRToBL,                                         // UART4_IRQHandler               @0x0000_0110    
    DispatchISRToBL,                                         // UART5_IRQHandler               @0x0000_0114    
    DispatchISRToBL,                                         // TIM6_DAC_IRQHandler            @0x0000_0118       
    DispatchISRToBL,                                         // TIM7_IRQHandler                @0x0000_011C 
    DispatchISRToBL,                                         // DMA2_Stream0_IRQHandler        @0x0000_0120    
    DispatchISRToBL,                                         // DMA2_Stream1_IRQHandler        @0x0000_0124    
    DispatchISRToBL,                                         // DMA2_Stream2_IRQHandler        @0x0000_0128    
    DispatchISRToBL,                                         // DMA2_Stream3_IRQHandler        @0x0000_012C    
    DispatchISRToBL,                                         // DMA2_Stream4_IRQHandler        @0x0000_0130    
    DispatchISRToBL,                                         // ETH_IRQHandler                 @0x0000_0134    
    DispatchISRToBL,                                         // ETH_WKUP_IRQHandler            @0x0000_0138           
    DispatchISRToBL,                                         // CAN2_TX_IRQHandler             @0x0000_013C           
    DispatchISRToBL,                                         // CAN2_RX0_IRQHandler            @0x0000_0140           
    DispatchISRToBL,                                         // CAN2_RX1_IRQHandler            @0x0000_0144           
    DispatchISRToBL,                                         // CAN2_SCE_IRQHandler            @0x0000_0148           
    DispatchISRToBL,                                         // OTG_FS_IRQHandler              @0x0000_014C    
    DispatchISRToBL,                                         // DMA2_Stream5_IRQHandler        @0x0000_0150    
    DispatchISRToBL,                                         // DMA2_Stream6_IRQHandler        @0x0000_0154    
    DispatchISRToBL,                                         // DMA2_Stream7_IRQHandler        @0x0000_0158    
    DispatchISRToBL,                                         // USART6_IRQHandler              @0x0000_015C     
    DispatchISRToBL,                                         // I2C3_EV_IRQHandler             @0x0000_0160           
    DispatchISRToBL,                                         // I2C3_ER_IRQHandler             @0x0000_0164           
    DispatchISRToBL,                                         // OTG_HS_EP1_OUT_IRQHandler      @0x0000_0168    
    DispatchISRToBL,                                         // OTG_HS_EP1_IN_IRQHandler       @0x0000_016C    
    DispatchISRToBL,                                         // OTG_HS_WKUP_IRQHandler         @0x0000_0170            
    DispatchISRToBL,                                         // OTG_HS_IRQHandler              @0x0000_0174    
    DispatchISRToBL,                                         // DCMI_IRQHandler                @0x0000_0178    
    DispatchISRToBL,                                         // CRYP_IRQHandler                @0x0000_017C    
    DispatchISRToBL,                                         // HASH_RNG_IRQHandler            @0x0000_0180 
    DispatchISRToBL,                                         // FPU_IRQHandler                 @0x0000_0184    
    DispatchISRToBL,                                         // UART7_IRQHandler               @0x0000_0188           
    DispatchISRToBL,                                         // UART8_IRQHandler               @0x0000_018C    
    DispatchISRToBL,                                         // SPI4_IRQHandler                @0x0000_0190    
    DispatchISRToBL,                                         // SPI5_IRQHandler                @0x0000_0194            
    DispatchISRToBL,                                         // SPI6_IRQHandler                @0x0000_0198    
    DispatchISRToBL,                                         // SAI1_IRQHandler                @0x0000_019C    
    DispatchISRToBL,                                         // LCD_TFTIRQHandler              @0x0000_01A0    
    DispatchISRToBL,                                         // LCD_TFTIRQErrorHandler         @0x0000_01A4 
    DispatchISRToBL,                                         // DMA2D_IRQHandler               @0x0000_01A8          
};    

SECTION_FLASH_ISRVECTORS_BL
IVTVector g_pfnVectorsBLStubs[IVTENTRYCT] = {0};  // this has a NOLOAD directive in the lcf, so it won't be part of the image!
